"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getUserById = void 0;
const db_1 = require("@b/db");
const console_1 = require("@b/utils/console");
const getUserById = async (id) => {
    const user = await db_1.models.user.findOne({
        where: { id },
        include: [
            {
                model: db_1.models.role,
                as: "role",
                attributes: ["id", "name"],
                include: [
                    {
                        model: db_1.models.permission,
                        as: "permissions",
                        through: { attributes: [] },
                        attributes: ["id", "name"],
                    },
                ],
            },
            {
                model: db_1.models.twoFactor,
                as: "twoFactor",
                attributes: ["type", "enabled"],
            },
            {
                model: db_1.models.kycApplication,
                as: "kyc",
                attributes: ["status"],
                include: [
                    {
                        model: db_1.models.kycLevel,
                        as: "level",
                        attributes: ["id", "name", "level", "features"],
                        paranoid: false, // kycLevel doesn't have soft deletes
                    },
                ],
            },
            {
                model: db_1.models.author,
                as: "author",
                attributes: ["id", "status"],
            },
            {
                model: db_1.models.providerUser,
                as: "providers",
                attributes: ["provider", "providerUserId"],
            },
        ],
        attributes: { exclude: ["password"] },
    });
    if (!user) {
        throw new Error("User not found");
    }
    // Convert to plain object
    const plainUser = user.get({ plain: true });
    // Set the user's KYC level and parse KYC features
    let featureAccess = [];
    // Only process feature access if KYC is approved and has a valid level
    if (plainUser.kyc && plainUser.kyc.status === "APPROVED" && plainUser.kyc.level) {
        plainUser.kycLevel = plainUser.kyc.level.level;
        try {
            if (plainUser.kyc.level.features) {
                // Features field may be null or already an array (rare); handle both
                if (typeof plainUser.kyc.level.features === "string") {
                    featureAccess = JSON.parse(plainUser.kyc.level.features);
                }
                else if (Array.isArray(plainUser.kyc.level.features)) {
                    featureAccess = plainUser.kyc.level.features;
                }
                // Ensure we have a valid array
                if (!Array.isArray(featureAccess)) {
                    featureAccess = [];
                }
            }
        }
        catch (err) {
            // Parsing failed or malformed; fallback to empty array
            console_1.logger.error("USER", "Error parsing KYC level features", err);
            featureAccess = [];
        }
    }
    else {
        // If KYC is not approved or missing level, no feature access
        plainUser.kycLevel = 0;
    }
    // Expose features as a top-level field for easy access
    plainUser.featureAccess = featureAccess;
    return plainUser;
};
exports.getUserById = getUserById;
