"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const utils_1 = require("./utils");
const db_1 = require("@b/db");
const emails_1 = require("@b/utils/emails");
const console_1 = require("@b/utils/console");
exports.metadata = {
    summary: "Verifies a Stripe payment intent",
    description: "Confirms a completed Stripe payment intent and creates the corresponding wallet transaction",
    operationId: "verifyStripePaymentIntent",
    tags: ["Finance", "Deposit"],
    requiresAuth: true,
    logModule: "STRIPE_DEPOSIT",
    logTitle: "Verify Stripe payment intent",
    parameters: [
        {
            index: 0,
            name: "intentId",
            in: "query",
            description: "Stripe payment intent ID (pi_xxxxx)",
            required: true,
            schema: { type: "string" },
        },
    ],
    responses: {
        200: {
            description: "Payment intent verified successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            transaction: {
                                type: "object",
                                description: "Created transaction object"
                            },
                            balance: {
                                type: "number",
                                description: "Updated wallet balance"
                            },
                            currency: {
                                type: "string",
                                description: "Transaction currency"
                            },
                            method: {
                                type: "string",
                                description: "Payment method"
                            },
                            status: {
                                type: "string",
                                description: "Verification status"
                            }
                        }
                    }
                }
            }
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Payment Intent"),
        500: query_1.serverErrorResponse,
    },
};
exports.default = async (data) => {
    const { user, query, ctx } = data;
    if (!user)
        throw new Error("User not authenticated");
    const { intentId } = query;
    const stripe = (0, utils_1.useStripe)();
    try {
        // 1. Retrieve the Payment Intent from Stripe
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Retrieving Stripe payment intent");
        const paymentIntent = await stripe.paymentIntents.retrieve(intentId);
        // 2. Check if payment succeeded
        if (paymentIntent.status !== 'succeeded') {
            throw new Error(`Payment intent status: ${paymentIntent.status}`);
        }
        // 3. Check for existing transaction to prevent duplicates
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Checking for duplicate transaction");
        const existingTransaction = await db_1.models.transaction.findOne({
            where: { referenceId: intentId },
        });
        if (existingTransaction) {
            ctx === null || ctx === void 0 ? void 0 : ctx.success("Stripe deposit completed successfully");
            return {
                transaction: existingTransaction,
                status: 'already_processed',
                message: 'Transaction already exists'
            };
        }
        // 4. Calculate amounts (same logic as checkout sessions)
        const totalAmount = paymentIntent.amount / 100; // Convert from cents
        const currency = paymentIntent.currency.toUpperCase();
        // Get gateway to calculate fees
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Fetching payment gateway configuration");
        const gateway = await db_1.models.depositGateway.findOne({
            where: { alias: "stripe", status: true },
        });
        if (!gateway)
            throw new Error("Stripe gateway not found");
        const { fixedFee, percentageFee } = gateway;
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Calculating fees");
        const fee = (totalAmount * (percentageFee || 0)) / 100 + (fixedFee || 0);
        const depositAmount = totalAmount - fee;
        // 5. Find or create user wallet
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Finding or creating wallet");
        let wallet = await db_1.models.wallet.findOne({
            where: { userId: user.id, currency, type: "FIAT" },
        });
        if (!wallet) {
            ctx === null || ctx === void 0 ? void 0 : ctx.step("Creating new wallet");
            wallet = await db_1.models.wallet.create({
                userId: user.id,
                currency,
                type: "FIAT",
            });
        }
        // Get currency data for precision
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Validating currency");
        const currencyData = await db_1.models.currency.findOne({
            where: { id: wallet.currency },
        });
        if (!currencyData) {
            ctx === null || ctx === void 0 ? void 0 : ctx.fail("Currency not found");
            throw new Error("Currency not found");
        }
        let newBalance = wallet.balance + depositAmount;
        newBalance = parseFloat(newBalance.toFixed(currencyData.precision || 2));
        // 6. Create transaction and update wallet (use database transaction)
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Starting database transaction");
        const result = await db_1.sequelize.transaction(async (t) => {
            // Create transaction record
            ctx === null || ctx === void 0 ? void 0 : ctx.step("Creating transaction record");
            const newTransaction = await db_1.models.transaction.create({
                userId: user.id,
                walletId: wallet.id,
                type: "DEPOSIT",
                amount: depositAmount,
                fee,
                referenceId: intentId,
                status: "COMPLETED",
                metadata: JSON.stringify({
                    method: "STRIPE_PAYMENT_INTENT",
                    paymentIntentId: intentId,
                    stripeAmount: totalAmount,
                }),
                description: `Stripe payment intent deposit of ${depositAmount} ${currency}`,
            }, { transaction: t });
            // Update wallet balance
            ctx === null || ctx === void 0 ? void 0 : ctx.step("Updating wallet balance");
            await db_1.models.wallet.update({ balance: newBalance }, { where: { id: wallet.id }, transaction: t });
            // Record admin profit if there's a fee
            if (fee > 0) {
                ctx === null || ctx === void 0 ? void 0 : ctx.step("Recording admin profit");
                await db_1.models.adminProfit.create({
                    amount: fee,
                    currency: wallet.currency,
                    type: "DEPOSIT",
                    transactionId: newTransaction.id,
                    description: `Admin profit from Stripe payment intent fee of ${fee} ${currency} for user (${user.id})`,
                }, { transaction: t });
            }
            return newTransaction;
        });
        // 7. Send email notification
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Fetching user account");
        const userPk = await db_1.models.user.findByPk(user.id);
        try {
            ctx === null || ctx === void 0 ? void 0 : ctx.step("Sending notification email");
            await (0, emails_1.sendFiatTransactionEmail)(userPk, result, currency, newBalance);
        }
        catch (error) {
            console_1.logger.error("STRIPE", "Error sending email", error);
        }
        return {
            transaction: result,
            balance: newBalance,
            currency,
            method: "Stripe Payment Intent",
            status: "succeeded"
        };
    }
    catch (error) {
        throw new Error(`Error verifying payment intent: ${error.message}`);
    }
};
