"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const console_1 = require("@b/utils/console");
const emails_1 = require("@b/utils/emails");
const db_1 = require("@b/db");
const utils_1 = require("./utils");
exports.metadata = {
    summary: "Verifies an Authorize.Net transaction",
    description: "Confirms the validity of an Authorize.Net transaction by its reference ID, ensuring the transaction is authenticated and processing the deposit.",
    operationId: "verifyAuthorizeNetTransaction",
    tags: ["Finance", "Deposit"],
    requiresAuth: true,
    logModule: "AUTHORIZENET_DEPOSIT",
    logTitle: "Verify Authorize.Net transaction",
    parameters: [
        {
            name: "referenceId",
            in: "query",
            description: "The transaction reference ID",
            required: true,
            schema: {
                type: "string",
            },
        },
    ],
    responses: {
        200: {
            description: "Transaction verified successfully. Returns the transaction details and updated wallet balance.",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            status: {
                                type: "boolean",
                                description: "Indicates if the request was successful",
                            },
                            statusCode: {
                                type: "number",
                                description: "HTTP status code",
                                example: 200,
                            },
                            data: {
                                type: "object",
                                properties: {
                                    transactionId: {
                                        type: "string",
                                        description: "Transaction ID",
                                    },
                                    status: {
                                        type: "string",
                                        description: "Transaction status",
                                    },
                                    amount: {
                                        type: "number",
                                        description: "Transaction amount",
                                    },
                                    currency: {
                                        type: "string",
                                        description: "Currency code",
                                    },
                                },
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Authorize.Net"),
        500: query_1.serverErrorResponse,
    },
};
exports.default = async (data) => {
    const { user, query, ctx } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id))
        throw new Error("User not authenticated");
    ctx === null || ctx === void 0 ? void 0 : ctx.step("Fetching user account");
    const userPk = await db_1.models.user.findByPk(user.id);
    if (!userPk)
        throw new Error("User not found");
    const { referenceId } = query;
    if (!referenceId) {
        throw new Error("Reference ID is required");
    }
    // Find the transaction
    const transaction = await db_1.models.transaction.findOne({
        where: { referenceId: referenceId, userId: user.id },
    });
    if (!transaction) {
        throw new Error("Transaction not found");
    }
    if (transaction.status === "COMPLETED") {
        return {
            status: true,
            statusCode: 200,
            data: {
                transactionId: transaction.id,
                status: transaction.status,
                amount: transaction.amount,
                currency: JSON.parse(transaction.metadata || "{}").currency,
                message: "Transaction already completed",
            },
        };
    }
    if (transaction.status !== "PENDING") {
        throw new Error(`Transaction is in ${transaction.status} state and cannot be verified`);
    }
    try {
        const config = (0, utils_1.getAuthorizeNetConfig)();
        const metadata = JSON.parse(transaction.metadata || "{}");
        const currency = metadata.currency;
        // Get transaction details from Authorize.Net
        // Note: Authorize.Net doesn't have a direct transaction details API for hosted payments
        // We'll mark the transaction as completed based on the return from hosted payment
        // In a real implementation, you might want to use webhooks for more reliable verification
        const authorizeNetGateway = await db_1.models.depositGateway.findOne({
            where: { name: "AUTHORIZENET" },
        });
        if (!authorizeNetGateway) {
            throw new Error("Authorize.Net gateway not found");
        }
        // Retrieve the user's wallet
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Finding or creating wallet");
        let wallet = await db_1.models.wallet.findOne({
            where: { userId: user.id, currency: currency },
        });
        if (!wallet) {
            ctx === null || ctx === void 0 ? void 0 : ctx.step("Creating new wallet");
            wallet = await db_1.models.wallet.create({
                userId: user.id,
                currency: currency,
                type: "FIAT",
            });
        }
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Validating currency");
        const currencyData = await db_1.models.currency.findOne({
            where: { id: wallet.currency },
        });
        if (!currencyData) {
            ctx === null || ctx === void 0 ? void 0 : ctx.fail("Currency not found");
            throw new Error("Currency not found");
        }
        const depositAmount = transaction.amount;
        const feeAmount = transaction.fee || 0;
        let newBalance = Number(wallet.balance);
        newBalance += depositAmount;
        newBalance = Number(newBalance.toFixed(currencyData.precision || 2));
        // Start a transaction to update the transaction record and wallet balance
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Starting database transaction");
        const result = await db_1.sequelize.transaction(async (dbTransaction) => {
            // Update transaction status
            await db_1.models.transaction.update({
                status: "COMPLETED",
                description: `Deposit of ${depositAmount} ${currency} to ${userPk.firstName} ${userPk.lastName} wallet by Authorize.Net.`,
            }, {
                where: { id: transaction.id },
                transaction: dbTransaction,
            });
            // Update the wallet's balance
            ctx === null || ctx === void 0 ? void 0 : ctx.step("Updating wallet balance");
            await db_1.models.wallet.update({
                balance: newBalance,
            }, {
                where: { id: wallet.id },
                transaction: dbTransaction,
            });
            // Record admin profit if there's a fee
            if (feeAmount > 0) {
                await db_1.models.adminProfit.create({
                    amount: feeAmount,
                    currency: wallet.currency,
                    type: "DEPOSIT",
                    description: `Authorize.Net deposit fee for transaction ${referenceId}`,
                    transactionId: transaction.id,
                }, { transaction: dbTransaction });
            }
            return transaction;
        });
        // Send confirmation email
        try {
            ctx === null || ctx === void 0 ? void 0 : ctx.step("Sending notification email");
            await (0, emails_1.sendFiatTransactionEmail)(userPk, {
                ...transaction.toJSON(),
                status: "COMPLETED",
            }, currency, newBalance);
        }
        catch (emailError) {
            console_1.logger.error("AUTHORIZENET", "Failed to send transaction email", emailError);
            // Don't throw error for email failure
        }
        return {
            status: true,
            statusCode: 200,
            data: {
                transactionId: transaction.id,
                status: "COMPLETED",
                amount: depositAmount,
                currency: currency,
                fee: feeAmount,
                newBalance: newBalance,
                referenceId: referenceId,
            },
        };
    }
    catch (error) {
        console_1.logger.error("AUTHORIZENET", "Transaction verification error", error);
        // Update transaction status to failed
        await db_1.models.transaction.update({
            status: "FAILED",
            description: `Failed Authorize.Net deposit: ${error instanceof Error ? error.message : "Unknown error"}`,
        }, {
            where: { id: transaction.id },
        });
        throw new Error(error instanceof Error ? error.message : "Failed to verify Authorize.Net transaction");
    }
};
