"use strict";
/**
 * GET /api/admin/finance/binary/settings/presets
 *
 * Retrieves all available binary settings presets.
 * Presets provide pre-configured settings for different risk profiles.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const query_1 = require("@b/utils/query");
const utils_1 = require("./utils");
exports.metadata = {
    summary: "Get Binary Settings Presets",
    operationId: "getBinarySettingsPresets",
    tags: ["Admin", "Binary", "Settings"],
    description: "Retrieves all available binary trading settings presets with their configurations and risk profiles.",
    responses: {
        200: {
            description: "Presets retrieved successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            presets: {
                                type: "array",
                                items: {
                                    type: "object",
                                    properties: {
                                        id: { type: "string" },
                                        name: { type: "string" },
                                        description: { type: "string" },
                                        riskLevel: { type: "string", enum: ["low", "medium", "high"] },
                                        features: { type: "array", items: { type: "string" } },
                                        settings: { type: "object" },
                                        validation: { type: "object" },
                                    },
                                },
                            },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        500: query_1.serverErrorResponse,
    },
    requiresAuth: true,
    permission: "access.binary.settings",
};
exports.default = async (data) => {
    const { user, ctx } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        throw new Error("Unauthorized");
    }
    ctx === null || ctx === void 0 ? void 0 : ctx.step("Building presets information");
    const presets = [
        {
            id: "conservative",
            name: "Conservative",
            description: "Low-risk configuration with tight limits and lower profit percentages. Suitable for new platforms or risk-averse operators.",
            riskLevel: "low",
            features: [
                "Only RISE_FALL enabled by default",
                "Tight barrier distances (0.1% - 0.3%)",
                "Lower max amounts ($10 - $500)",
                "Profit range: 50-70%",
                "5 second cooldown between orders",
                "Daily loss limit enabled",
            ],
            settings: utils_1.BINARY_PRESETS.conservative,
            validation: (0, utils_1.validateBinarySettings)(utils_1.BINARY_PRESETS.conservative),
        },
        {
            id: "balanced",
            name: "Balanced",
            description: "Default balanced configuration suitable for most platforms. Moderate risk with reasonable profit margins.",
            riskLevel: "medium",
            features: [
                "Only RISE_FALL enabled by default",
                "Moderate barrier distances (0.1% - 0.5%)",
                "Standard amounts ($1 - $1,000)",
                "Profit range: 65-85%",
                "No cooldown between orders",
                "Configurable risk limits",
            ],
            settings: utils_1.BINARY_PRESETS.balanced,
            validation: (0, utils_1.validateBinarySettings)(utils_1.BINARY_PRESETS.balanced),
        },
        {
            id: "aggressive",
            name: "Aggressive",
            description: "Higher-risk configuration with more features enabled and higher profit potential. Suitable for established platforms with good risk management.",
            riskLevel: "high",
            features: [
                "RISE_FALL and HIGHER_LOWER enabled",
                "Wider barrier distances (0.1% - 1%)",
                "Higher max amounts ($1 - $10,000)",
                "Profit range: 55-90%",
                "No cooldown between orders",
                "Relaxed risk limits",
                "Extended duration options",
            ],
            settings: utils_1.BINARY_PRESETS.aggressive,
            validation: (0, utils_1.validateBinarySettings)(utils_1.BINARY_PRESETS.aggressive),
        },
    ];
    ctx === null || ctx === void 0 ? void 0 : ctx.success("Presets retrieved successfully");
    return {
        presets,
    };
};
